// Generated by CoffeeScript 2.4.1
  //Copyright (c) 2013-2019 Hypothes.is Project and contributors

  //Redistribution and use in source and binary forms, with or without
  //modification, are permitted provided that the following conditions are met:

  //1. Redistributions of source code must retain the above copyright notice, this
  //   list of conditions and the following disclaimer.
  //2. Redistributions in binary form must reproduce the above copyright notice,
  //   this list of conditions and the following disclaimer in the documentation
  //   and/or other materials provided with the distribution.

  //THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  //ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  //WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  //DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
  //ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  //(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  //LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  //ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  //(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  //SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  //This script is a modified copy of https://github.com/hypothesis/client/blob/master/src/annotator/anchoring/range.coffee
  import { Util } from './util'
  import * as $ from 'jquery'

  // Public: Determines the type of Range of the provided object and returns
  // a suitable Range instance.

  // r - A range Object.

  // Examples

  //   selection = window.getSelection()
  //   Range.sniff(selection.getRangeAt(0))
  //   # => Returns a BrowserRange instance.

  // Returns a Range object or false.

  export var Range = {};

  Range.sniff = function(r) {
    if (r.commonAncestorContainer != null) {
      return new exports.BrowserRange(r);
    } else if (typeof r.start === "string") {
      return new exports.SerializedRange(r);
    } else if (r.start && typeof r.start === "object") {
      return new exports.NormalizedRange(r);
    } else {
      console.error(_t("Could not sniff range type"));
      return false;
    }
  };

  // Public: Finds an Element Node using an XPath relative to the document root.

  // If the document is served as application/xhtml+xml it will try and resolve
  // any namespaces within the XPath.

  // xpath - An XPath String to query.

  // Examples

  //   node = Range.nodeFromXPath('/html/body/div/p[2]')
  //   if node
  //     # Do something with the node.

  // Returns the Node if found otherwise null.
  Range.nodeFromXPath = function(xpath, root = document) {
    var customResolver, evaluateXPath, namespace, node, segment;
    evaluateXPath = function(xp, nsResolver = null) {
      var exception;
      try {
        return document.evaluate('.' + xp, root, nsResolver, XPathResult.FIRST_ORDERED_NODE_TYPE, null).singleNodeValue;
      } catch (error) {
        exception = error;
        // There are cases when the evaluation fails, because the
        // HTML documents contains nodes with invalid names,
        // for example tags with equal signs in them, or something like that.
        // In these cases, the XPath expressions will have these abominations,
        // too, and then they can not be evaluated.
        // In these cases, we get an XPathException, with error code 52.
        // See http://www.w3.org/TR/DOM-Level-3-XPath/xpath.html#XPathException
        // This does not necessarily make any sense, but this what we see
        // happening.
        console.log("XPath evaluation failed.");
        console.log("Trying fallback...");
        // We have a an 'evaluator' for the really simple expressions that
        // should work for the simple expressions we generate.
        return Util.nodeFromXPath(xp, root);
      }
    };
    if (!$.isXMLDoc(document.documentElement)) {
      return evaluateXPath(xpath);
    } else {
      // We're in an XML document, create a namespace resolver function to try
      // and resolve any namespaces in the current document.
      // https://developer.mozilla.org/en/DOM/document.createNSResolver
      customResolver = document.createNSResolver(document.ownerDocument === null ? document.documentElement : document.ownerDocument.documentElement);
      node = evaluateXPath(xpath, customResolver);
      if (!node) {
        // If the previous search failed to find a node then we must try to
        // provide a custom namespace resolver to take into account the default
        // namespace. We also prefix all node names with a custom xhtml namespace
        // eg. 'div' => 'xhtml:div'.
        xpath = ((function() {
          var i, len, ref, results;
          ref = xpath.split('/');
          results = [];
          for (i = 0, len = ref.length; i < len; i++) {
            segment = ref[i];
            if (segment && segment.indexOf(':') === -1) {
              results.push(segment.replace(/^([a-z]+)/, 'xhtml:$1'));
            } else {
              results.push(segment);
            }
          }
          return results;
        })()).join('/');
        // Find the default document namespace.
        namespace = document.lookupNamespaceURI(null);
        // Try and resolve the namespace, first seeing if it is an xhtml node
        // otherwise check the head attributes.
        customResolver = function(ns) {
          if (ns === 'xhtml') {
            return namespace;
          } else {
            return document.documentElement.getAttribute('xmlns:' + ns);
          }
        };
        node = evaluateXPath(xpath, customResolver);
      }
      return node;
    }
  };

  // Public: Creates a wrapper around a range object obtained from a DOMSelection.
  export class BrowserRange {
    // Public: Creates an instance of BrowserRange.

    // object - A range object obtained via DOMSelection#getRangeAt().

    // Examples

    //   selection = window.getSelection()
    //   range = new Range.BrowserRange(selection.getRangeAt(0))

    // Returns an instance of BrowserRange.
    constructor(obj) {
      this.commonAncestorContainer = obj.commonAncestorContainer;
      this.startContainer = obj.startContainer;
      this.startOffset = obj.startOffset;
      this.endContainer = obj.endContainer;
      this.endOffset = obj.endOffset;
    }

    // Public: normalize works around the fact that browsers don't generate
    // ranges/selections in a consistent manner. Some (Safari) will create
    // ranges that have (say) a textNode startContainer and elementNode
    // endContainer. Others (Firefox) seem to only ever generate
    // textNode/textNode or elementNode/elementNode pairs.

    // Returns an instance of Range.NormalizedRange
    normalize(root) {
      var n, node, nr, r;
      if (this.tainted) {
        console.error(_t("You may only call normalize() once on a BrowserRange!"));
        return false;
      } else {
        this.tainted = true;
      }
      r = {};
      // Look at the start
      if (this.startContainer.nodeType === Node.ELEMENT_NODE) {
        // We are dealing with element nodes
        r.start = Util.getFirstTextNodeNotBefore(this.startContainer.childNodes[this.startOffset]);
        r.startOffset = 0;
      } else {
        // We are dealing with simple text nodes
        r.start = this.startContainer;
        r.startOffset = this.startOffset;
      }
      // Look at the end
      if (this.endContainer.nodeType === Node.ELEMENT_NODE) {
        // Get specified node.
        node = this.endContainer.childNodes[this.endOffset];
        if (node != null) {
          // Look for a text node either at the immediate beginning of node
          n = node;
          while ((n != null) && (n.nodeType !== Node.TEXT_NODE)) {
            n = n.firstChild;
          }
          if (n != null) {
            r.end = n;
            r.endOffset = 0; // Did we find a text node at the start of this element?
          } // Does that node exist?
        }
        if (r.end == null) {

          // We need to find a text node in the previous sibling of the node at the
          // given offset, if one exists, or in the previous sibling of its container.
          if (this.endOffset) {
            node = this.endContainer.childNodes[this.endOffset - 1];
          } else {
            node = this.endContainer.previousSibling;
          }
          r.end = Util.getLastTextNodeUpTo(node);
          r.endOffset = r.end.nodeValue.length; // We are dealing with simple text nodes
        }
      } else {
        r.end = this.endContainer;
        r.endOffset = this.endOffset;
      }
      // We have collected the initial data.

      // Now let's start to slice & dice the text elements!
      nr = {};
      if (r.startOffset > 0) {
        // Do we really have to cut?
        if (r.start.nodeValue.length > r.startOffset) {
          // Yes. Cut.
          nr.start = r.start.splitText(r.startOffset);
        } else {
          // Avoid splitting off zero-length pieces.
          nr.start = r.start.nextSibling;
        }
      } else {
        nr.start = r.start;
      }
      // is the whole selection inside one text element ?
      if (r.start === r.end) {
        if (nr.start.nodeValue.length > (r.endOffset - r.startOffset)) {
          nr.start.splitText(r.endOffset - r.startOffset);
        }
        nr.end = nr.start; // no, the end of the selection is in a separate text element
      } else {
        // does the end need to be cut?
        if (r.end.nodeValue.length > r.endOffset) {
          r.end.splitText(r.endOffset);
        }
        nr.end = r.end;
      }
      // Make sure the common ancestor is an element node.
      nr.commonAncestor = this.commonAncestorContainer;
      while (nr.commonAncestor.nodeType !== Node.ELEMENT_NODE) {
        nr.commonAncestor = nr.commonAncestor.parentNode;
      }
      return new exports.NormalizedRange(nr);
    }

    // Public: Creates a range suitable for storage.

    // root           - A root Element from which to anchor the serialisation.
    // ignoreSelector - A selector String of elements to ignore. For example
    //                  elements injected by the annotator.

    // Returns an instance of SerializedRange.
    serialize(root, ignoreSelector) {
      return this.normalize(root).serialize(root, ignoreSelector);
    }

  };

  // Public: A normalised range is most commonly used throughout the annotator.
  // its the result of a deserialised SerializedRange or a BrowserRange with
  // out browser inconsistencies.
  export class NormalizedRange {
    // Public: Creates an instance of a NormalizedRange.

    // This is usually created by calling the .normalize() method on one of the
    // other Range classes rather than manually.

    // obj - An Object literal. Should have the following properties.
    //       commonAncestor: A Element that encompasses both the start and end nodes
    //       start:          The first TextNode in the range.
    //       end             The last TextNode in the range.

    // Returns an instance of NormalizedRange.
    constructor(obj) {
      this.commonAncestor = obj.commonAncestor;
      this.start = obj.start;
      this.end = obj.end;
    }

    // Public: For API consistency.

    // Returns itself.
    normalize(root) {
      return this;
    }

    // Public: Limits the nodes within the NormalizedRange to those contained
    // withing the bounds parameter. It returns an updated range with all
    // properties updated. NOTE: Method returns null if all nodes fall outside
    // of the bounds.

    // bounds - An Element to limit the range to.

    // Returns updated self or null.
    limit(bounds) {
      var i, len, nodes, parent, ref, startParents;
      nodes = $.grep(this.textNodes(), function(node) {
        return node.parentNode === bounds || $.contains(bounds, node.parentNode);
      });
      if (!nodes.length) {
        return null;
      }
      this.start = nodes[0];
      this.end = nodes[nodes.length - 1];
      startParents = $(this.start).parents();
      ref = $(this.end).parents();
      for (i = 0, len = ref.length; i < len; i++) {
        parent = ref[i];
        if (startParents.index(parent) !== -1) {
          this.commonAncestor = parent;
          break;
        }
      }
      return this;
    }

    // Convert this range into an object consisting of two pairs of (xpath,
    // character offset), which can be easily stored in a database.

    // root -           The root Element relative to which XPaths should be calculated
    // ignoreSelector - A selector String of elements to ignore. For example
    //                  elements injected by the annotator.

    // Returns an instance of SerializedRange.
    serialize(root, ignoreSelector) {
      var end, serialization, start;
      serialization = function(node, isEnd) {
        var i, len, n, nodes, offset, origParent, textNodes, xpath;
        if (ignoreSelector) {
          origParent = $(node).parents(`:not(${ignoreSelector})`).eq(0);
        } else {
          origParent = $(node).parent();
        }
        xpath = Util.xpathFromNode(origParent, root)[0];
        textNodes = Util.getTextNodes(origParent);
        // Calculate real offset as the combined length of all the
        // preceding textNode siblings. We include the length of the
        // node if it's the end node.
        nodes = textNodes.slice(0, textNodes.index(node));
        offset = 0;
        for (i = 0, len = nodes.length; i < len; i++) {
          n = nodes[i];
          offset += n.nodeValue.length;
        }
        if (isEnd) {
          return [xpath, offset + node.nodeValue.length];
        } else {
          return [xpath, offset];
        }
      };
      start = serialization(this.start);
      end = serialization(this.end, true);
      return new exports.SerializedRange({
        // XPath strings
        start: start[0],
        end: end[0],
        // Character offsets (integer)
        startOffset: start[1],
        endOffset: end[1]
      });
    }

    // Public: Creates a concatenated String of the contents of all the text nodes
    // within the range.

    // Returns a String.
    text() {
      var node;
      return ((function() {
        var i, len, ref, results;
        ref = this.textNodes();
        results = [];
        for (i = 0, len = ref.length; i < len; i++) {
          node = ref[i];
          results.push(node.nodeValue);
        }
        return results;
      }).call(this)).join('');
    }

    // Public: Fetches only the text nodes within th range.

    // Returns an Array of TextNode instances.
    textNodes() {
      var end, start, textNodes;
      textNodes = Util.getTextNodes($(this.commonAncestor));
      [start, end] = [textNodes.index(this.start), textNodes.index(this.end)];
      // Return the textNodes that fall between the start and end indexes.
      return $.makeArray(textNodes.slice(start, +end + 1 || 9e9));
    }

    // Public: Converts the Normalized range to a native browser range.

    // See: https://developer.mozilla.org/en/DOM/range

    // Examples

    //   selection = window.getSelection()
    //   selection.removeAllRanges()
    //   selection.addRange(normedRange.toRange())

    // Returns a Range object.
    toRange() {
      var range;
      range = document.createRange();
      range.setStartBefore(this.start);
      range.setEndAfter(this.end);
      return range;
    }

  };

  // Public: A range suitable for storing in local storage or serializing to JSON.
  export class SerializedRange {
    // Public: Creates a SerializedRange

    // obj - The stored object. It should have the following properties.
    //       start:       An xpath to the Element containing the first TextNode
    //                    relative to the root Element.
    //       startOffset: The offset to the start of the selection from obj.start.
    //       end:         An xpath to the Element containing the last TextNode
    //                    relative to the root Element.
    //       startOffset: The offset to the end of the selection from obj.end.

    // Returns an instance of SerializedRange
    constructor(obj) {
      this.start = obj.start;
      this.startOffset = obj.startOffset;
      this.end = obj.end;
      this.endOffset = obj.endOffset;
    }

    // Public: Creates a NormalizedRange.

    // root - The root Element from which the XPaths were generated.

    // Returns a NormalizedRange instance.
    normalize(root) {
      var contains, e, i, j, len, len1, length, node, p, range, ref, ref1, targetOffset, tn;
      range = {};
      ref = ['start', 'end'];
      for (i = 0, len = ref.length; i < len; i++) {
        p = ref[i];
        try {
          node = exports.Range.nodeFromXPath(this[p], root);
        } catch (error) {
          e = error;
          console.log(`Error while finding ${p} node: ${this[p]}: ` + e, e);
        }
        if (!node) {
          console.log(`Couldn't find ${p} node: ${this[p]}`);
          throw("Range not found error");
        }
        // Unfortunately, we *can't* guarantee only one textNode per
        // elementNode, so we have to walk along the element's textNodes until
        // the combined length of the textNodes to that point exceeds or
        // matches the value of the offset.
        length = 0;
        targetOffset = this[p + 'Offset'];
        // Range excludes its endpoint because it describes the boundary position.
        // Target the string index of the last character inside the range.
        if (p === 'end') {
          targetOffset--;
        }
        ref1 = Util.getTextNodes($(node));
        for (j = 0, len1 = ref1.length; j < len1; j++) {
          tn = ref1[j];
          if (length + tn.nodeValue.length > targetOffset) {
            range[p + 'Container'] = tn;
            range[p + 'Offset'] = this[p + 'Offset'] - length;
            break;
          } else {
            length += tn.nodeValue.length;
          }
        }
        // If we fall off the end of the for loop without having set
        // 'startOffset'/'endOffset', the element has shorter content than when
        // we annotated, so throw an error:
        if (range[p + 'Offset'] == null) {
          console.log(`Couldn't find offset ${this[p + 'Offset']} in element ${this[p]}`);
        }
      }
      // Here's an elegant next step...

      //   range.commonAncestorContainer = $(range.startContainer).parents().has(range.endContainer)[0]

      // ...but unfortunately Node.contains() is broken in Safari 5.1.5 (7534.55.3)
      // and presumably other earlier versions of WebKit. In particular, in a
      // document like

      //   <p>Hello</p>

      // the code

      //   p = document.getElementsByTagName('p')[0]
      //   p.contains(p.firstChild)

      // returns `false`. Yay.

      // So instead, we step through the parents from the bottom up and use
      // Node.compareDocumentPosition() to decide when to set the
      // commonAncestorContainer and bail out.
      // IE
      contains = document.compareDocumentPosition == null ? function(a, b) {
        return a.contains(b);
      // Everyone else
      } : function(a, b) {
        return a.compareDocumentPosition(b) & 16;
      };
      $(range.startContainer).parents().each(function() {
        if (contains(this, range.endContainer)) {
          range.commonAncestorContainer = this;
          return false;
        }
      });
      return new exports.BrowserRange(range).normalize(root);
    }

    // Public: Creates a range suitable for storage.

    // root           - A root Element from which to anchor the serialisation.
    // ignoreSelector - A selector String of elements to ignore. For example
    //                  elements injected by the annotator.

    // Returns an instance of SerializedRange.
    serialize(root, ignoreSelector) {
      return this.normalize(root).serialize(root, ignoreSelector);
    }

    // Public: Returns the range as an Object literal.
    toObject() {
      return {
        start: this.start,
        startOffset: this.startOffset,
        end: this.end,
        endOffset: this.endOffset
      };
    }

  };

