import * as $ from "jquery"
import { Util } from "./util.js"
import { TextRange } from "./pdf-text-range.js";

const PREFIX_LENGTH = 10;
const SUFFIX_LENGTH = 10;
const EXACT_WEIGHT = 0.5;
const PREFIX_WEIGHT = 0.25;
const SUFFIX_WEIGHT = 0.25;

export function findHTMLTextQuote(root, textQuote) {
    let bestMatch = null;
    let textNodes = Util.getTextNodes($(root));
    let websideString = Util.joinTextNodes(textNodes).data;
    let matches = [];
    let splittedExact = splitExact(textQuote.exact);
    matches = getIndicesOf(textQuote.exact, websideString);
    if (matches.length === 0) return null;
    for (let match of matches) {
        let prefix = websideString.substring(match.index - textQuote.prefix.length, match.index);
        let suffix = websideString.substring(match.index + textQuote.exact.length, match.index + textQuote.exact.length + textQuote.suffix.length);
        let distance = distanceOf(prefix, textQuote.prefix);
        distance = distance > textQuote.prefix.length ? textQuote.prefix.length : distance;
        let parity = (textQuote.prefix.length - distance) / textQuote.prefix.length;
        match.score += parity * PREFIX_WEIGHT;
        distance = distanceOf(suffix, textQuote.suffix);
        distance = distance > textQuote.suffix.length ? textQuote.suffix.length : distance;
        parity = (textQuote.suffix.length - distance) / textQuote.suffix.length;
        match.score += parity * SUFFIX_WEIGHT;
        if (bestMatch === null || match.score > bestMatch.score) {
            bestMatch = match;
        }

    }
    //best match found, now extract range selector
    if (bestMatch !== null) {
        let idx = 0;
        for (let node of textNodes) {
            idx += node.data.length;
            let startParent;
            if ((idx - 1) >= bestMatch.index) {
                startParent = node.parentElement;
                let parentsContent = Util.joinTextNodes(Util.getTextNodes($(startParent)));
                let index = parentsContent.textContent.indexOf(splittedExact[0]);
                let lastExactElement = startParent;
                let tmpStart = 0;
                for (let i = 1; i < splittedExact.length; i++) {
                    lastExactElement = Util.getNextTextElement(lastExactElement);
                    tmpStart = index;
                    index = 0;
                }
                if (lastExactElement.nodeType === Node.TEXT_NODE) {
                    lastExactElement = lastExactElement.parentElement;
                }
                let lastExact = splittedExact[splittedExact.length - 1];
                let endOffset;
                if ((endOffset = lastExactElement.textContent.indexOf(lastExact, index)) !== -1) {
                    index = index == 0 ? tmpStart : index;
                    return {
                        type: "RangeSelector",
                        startContainer: Util.xpathFromNode($(startParent), root)[0],
                        startOffset: index,
                        endContainer: Util.xpathFromNode($(lastExactElement), root)[0],
                        endOffset: endOffset + lastExact.length
                    };
                } else {
                    return null;
                }

            }
        }

    } else {
        return null;
    }

}
// find text quote in a PDF page
export function getBestMatchForPage(text, quote, hint) {
    let bestMatches = [];
    let bestScore = 0;
    if (quote.exact.length === 0) return null;

    let matches = []
    matches = getIndicesOf(quote.exact, text);
    if (matches.length === 0) return null;

    for(let match of matches) {
        const prefix = text.substring(match.index - quote.prefix.length, match.index);
        const suffix = text.substring(match.index + quote.exact.length, match.index + quote.exact.length + quote.suffix.length); 
        let distance = distanceOf(prefix, quote.prefix);
        distance = distance > quote.prefix.length ? quote.prefix.length : distance;
        let parity = (quote.prefix.length - distance) / quote.prefix.length;
        match.score += parity * PREFIX_WEIGHT;
        distance = distanceOf(suffix, quote.suffix);
        distance = distance > quote.suffix.length ? quote.suffix.length : distance;
        parity = (quote.suffix.length - distance) / quote.suffix.length;
        match.score += parity * SUFFIX_WEIGHT;
        if (bestScore === 0 || match.score >= bestScore) {
            bestScore = match.score;
        }
    }
    for(let match of matches) {
        if(match.score === bestScore) {
            bestMatches.push(match);
        }
    }
    if(bestMatches.length > 1) {
        let index = 0, distance = 0;
        for(let i = 0; i <= bestMatches.length; i++) {
            let tmp = Math.abs(bestMatches[i].index - hint);
            if ( i === 0) {
                index = i;
                distance = tmp;
            } else {
                if(distance < tmp) {
                    index = i;
                    distance = tmp;
                }
            }
        }
        return bestMatches[index];
    } else return bestMatches[0];
    

}


export function createTextQuote(range, rangeSelector) {
    let startElement = range.startContainer;
    let endElement = range.endContainer;
    let startNode, endNode;
    if (startElement.nodeType === Node.TEXT_NODE) {
        startNode = startElement;
        startElement = startNode.parentElement;
    }
    if (endElement.nodeType === Node.TEXT_NODE) {
        endNode = endElement;
        endElement = endNode.parentElement;
    }

    startNode = Util.joinTextNodes(Util.getTextNodes($(startElement)));
    endNode = Util.joinTextNodes(Util.getTextNodes($(endElement)));
    let prefix = "";
    let suffix = "";
    if (rangeSelector.startOffset < PREFIX_LENGTH) {
        const remaining = PREFIX_LENGTH - rangeSelector.startOffset;
        prefix = startNode.textContent.substr(0, rangeSelector.startOffset);
        const nodeBefore = Util.getPreviousTextElement(startElement);
        prefix = getPrefix(nodeBefore, remaining, prefix);
    } else {
        prefix = startNode.textContent.substr((rangeSelector.startOffset - PREFIX_LENGTH), PREFIX_LENGTH);
    }
    if (rangeSelector.endOffset + SUFFIX_LENGTH > endNode.textContent.length) {
        const remaining = SUFFIX_LENGTH - (endNode.textContent.length - rangeSelector.endOffset);
        suffix = endNode.textContent.substr(rangeSelector.endOffset);
        const nodeAfter = Util.getNextTextElement(endElement);
        suffix = getSuffix(nodeAfter, remaining, suffix);
    } else {
        suffix = endNode.textContent.substr(rangeSelector.endOffset, SUFFIX_LENGTH);
    }
    return new TextQuoteSelector("", suffix, prefix);

    //let textNodes = Util.getTextNodes(root);

}

function splitExact(exact) {
    let result = []
    let splitted = "";
    for (let i = 0; i < exact.length; i++) {
        if (exact[i] === '\n' || exact[i] === '\t') {
            if (splitted.length > 0) {
                result.push(splitted);
                splitted = "";
            }
        } else {
            splitted += exact[i];
        }
    }
    if (splitted.length > 0) {
        result.push(splitted);
    }
    return result;
}

function getIndicesOf(exact, text) {
    let exactLen = exact.length;
    if (exactLen == 0) {
        return [];
    }
    let startIndex = 0, index, indices = [];
    while ((index = text.indexOf(exact, startIndex)) > -1) {
        indices.push({ index: index, score: EXACT_WEIGHT });
        startIndex = index + exactLen;
    }
    return indices;
}

function getPrefix(textNode, remaining, prefix) {
    if (textNode !== null) {
        if (textNode.textContent.length >= remaining) {
            return (textNode.textContent.substr((textNode.textContent.length - remaining)) + prefix);
        } else {
            remaining = remaining - textNode.textContent.length;
            const nodeBefore = Util.getPreviousTextElement(textNode);
            return getPrefix(nodeBefore, remaining, textNode.textContent) + prefix;
        }
    } else {
        return "";
    }

}

function getSuffix(textNode, remaining, suffix) {
    if (textNode !== null) {
        if (textNode.textContent.length >= remaining) {
            return suffix + textNode.textContent.substr(0, remaining);
        } else {
            remaining = remaining - textNode.textContent.length;
            const nodeAfter = Util.getNextTextElement(textNode);
            return suffix + getSuffix(nodeAfter, remaining, textNode.textContent);
        }
    } else {
        return "";
    }

}
//Calculate Levenshtein distance between two strings
function distanceOf(a, b) {
    if (a.length === 0) return b.length;
    if (b.length === 0) return a.length;
    if (a[0] === b[0]) return distanceOf(a.substring(1), b.substring(1));
    let deletion = distanceOf(a.substring(1), b);
    let insertion = distanceOf(a, b.substring(1));
    let replacement = distanceOf(a.substring(1), b.substring(1));
    let min = (no1, no2, no3) => {
        return Math.min(no1, Math.min(no2, no3));
    }
    return 1 + min(deletion, insertion, replacement);
}


export class TextQuoteSelector {

    constructor(exact, suffix, prefix) {
        this.type = "TextQuoteSelector";
        this.exact = exact;
        this.suffix = suffix;
        this.prefix = prefix;
    }

}

/* ----------------------------------------------------------------------------------------------
Copyright (c) 2013-2019 Hypothes.is Project and contributors                                    |
                                                                                                
Redistribution and use in source and binary forms, with or without                              
modification, are permitted provided that the following conditions are met:                     
                                                                                                |
1. Redistributions of source code must retain the above copyright notice, this                  
   list of conditions and the following disclaimer.                                             
2. Redistributions in binary form must reproduce the above copyright notice,                    
   this list of conditions and the following disclaimer in the documentation                    |
   and/or other materials provided with the distribution.                                       
                                                                                                
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND                 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED                   |
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE                          
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR                 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES                  
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;                    |
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND                     
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT                      
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS                   
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                                    |
*/
export class TextQuoteAnchor {
    constructor(root, exact, context = {}) {
        this.root = root;//                                                                     |
        this.exact = exact;
        this.context = context;
    }
    //                                                                                          |
    /**
     * Create a `TextQuoteAnchor` from a range.
     *
     * Will throw if `range` does not contain any text nodes.                                   |
     *
     * @param {Element} root
     * @param {Range} range
     *                                                                                          |
     */
    static fromSelector(root, selector) {
        const prefix = selector.prefix, suffix = selector.suffix ;
        return new TextQuoteAnchor(root, selector.exact, { prefix, suffix });//                 |
    }

    static fromRange(root, range) {
        const text = /** @type {string} */ (root.textContent);
        const textRange = TextRange.fromRange(range).relativeTo(root);
        //                                                                                      |
        const start = textRange.start.offset;
        const end = textRange.end.offset;

        return new TextQuoteAnchor(root, text.slice(start, end), {                    
            prefix: text.slice(Math.max(0, start - PREFIX_LENGTH), start),
            suffix: text.slice(end, Math.min(text.length, end + SUFFIX_LENGTH))
        });
    }//                                                                                         |

    toSelector() {
        return new TextQuoteSelector(this.exact, this.context.suffix, this.context.prefix);//  |
    }

    toRange(options = {}) {
        return this.toPositionAnchor(options).toRange();//                                      |
    }

    toPositionAnchor(options = {}) {
        const text = /** @type {string} */ (this.root.textContent);//                           |
        const match = matchQuote(text, this.exact, {
            ...this.context,
            hint: options.hint,
        });//                                                                                   |
        if (!match) {
            throw new Error('Quote not found');
        }
        return new TextPositionAnchor(this.root, match.start, match.end);//                     |
    }//                                                                                         
}//                                                                                           
//-----------------------------------------------------------------------------------------------

